<?php 
header('Content-Type: application/json; charset=UTF-8');

// Retrieve parameters with null coalescing operator
$nid = $_GET['nid'] ?? null;
$dob = $_GET['dob'] ?? null;

// Validate parameters
if (!$nid || !$dob) {
    echo json_encode([
        "code" => 400,
        "error" => "NID and DOB parameters are required."
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

// Validate NID (must be numeric and minimum 10 digits)
if (!is_numeric($nid) || strlen($nid) < 10) {
    echo json_encode([
        "code" => 422,
        "error" => "Invalid NID format. It must be numeric and at least 10 digits."
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

// Validate DOB (must be a valid date in YYYY-MM-DD format)
$dateTime = DateTime::createFromFormat('Y-m-d', $dob);
if (!$dateTime || $dateTime->format('Y-m-d') !== $dob) {
    echo json_encode([
        "code" => 422,
        "error" => "Invalid DOB format. Use YYYY-MM-DD."
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

// Format the date of birth for API request
$dobFormatted = $dateTime->format("d/m/Y");

// Initialize cURL session
$curl = curl_init();

// Set cURL options
curl_setopt_array($curl, [
    CURLOPT_URL => "https://www.dis.gov.bd/SurveyForm/VerifyNID",
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_POST => true,
    CURLOPT_POSTFIELDS => json_encode([
        "NID" => $nid,
        "DOB" => $dobFormatted
    ]),
    CURLOPT_HTTPHEADER => [
        "Content-Type: application/json",
        "Origin: https://www.dis.gov.bd",
        "Referer: https://www.dis.gov.bd/SurveyForm/newFormregistration",
        "User-Agent: Mozilla/5.0 (Linux; Android 6.0; Nexus 5 Build/MRA58N) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/131.0.0.0 Mobile Safari/537.36",
        "X-Requested-With: XMLHttpRequest"
    ]
]);

$response = curl_exec($curl);

if (curl_errno($curl)) {
    echo json_encode([
        "code" => 500,
        "error" => "cURL Error: " . curl_error($curl)
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    curl_close($curl);
    exit;
}

$httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
curl_close($curl);

// Handle non-200 HTTP responses
if ($httpCode !== 200) {
    echo json_encode([
        "code" => $httpCode,
        "error" => "External API returned HTTP status code " . $httpCode
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

// Check if API response is valid JSON
$responseData = json_decode($response, true);

if (json_last_error() !== JSON_ERROR_NONE) {
    echo json_encode([
        "code" => 500,
        "error" => "Invalid JSON format. Unable to decode the response."
    ], JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

// Successfully return the decoded response
echo json_encode($responseData, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
?>
